#!/usr/bin/env python3

import pylibmodbus
import time

# Siemens LOGO! 8 Modbus TCP Configuration
PLC_IP = "192.168.1.103"  # Change this to your LOGO! 8 IP
PLC_PORT = 502  # Default Modbus TCP Port

# NQ Register Mapping
NQ_START_COIL = 0  # First NQ coil (adjust as needed)
NQ_TOTAL_COUNT = 1024  # Adjust based on the number of NQ outputs
CHUNK_SIZE = 5  # Reduce the number of coils read at a time

# NQ Holding Register Mapping (Adjust based on LOGO! 8 settings)
NQ_START_REGISTER = 64  # First NQ register (check Siemens LOGO! register map)



def main():
    """Read all NQ (Network Output) values from Siemens LOGO! 8 in chunks."""
    modbus = pylibmodbus.ModbusTcp(PLC_IP, PLC_PORT)

    try:
        if modbus.connect() == -1:
            print("❌ Failed to connect to Siemens LOGO! 8 PLC")
            return
        
        print("✅ Connected to Siemens LOGO! 8 PLC")

        # Read NQ outputs in smaller chunks
        for start in range(NQ_START_COIL, NQ_TOTAL_COUNT, CHUNK_SIZE):
            end = min(start + CHUNK_SIZE, NQ_TOTAL_COUNT)
            nq_coils = modbus.read_bits(start, end - start)

            if nq_coils is not None:
                for i, value in enumerate(nq_coils):
                    if ( value != 0 ):
                        print(f"🔹 NQ{start + i + 1} (Coil {start + i}) = {value}")
            else:
                print(f"⚠ Failed to read NQ outputs for Coils {start} to {end - 1}")

            time.sleep(0.1)  # Small delay to avoid overloading LOGO! 8
        # Read NQ outputs register
        for start in range(NQ_START_REGISTER, NQ_START_REGISTER + NQ_TOTAL_COUNT, CHUNK_SIZE):
            end = min(start + CHUNK_SIZE, NQ_START_REGISTER + NQ_TOTAL_COUNT)
            nq_registers = modbus.read_registers(start, end - start)

            if nq_registers is not None:
                for i, value in enumerate(nq_registers):
                    if ( value != 0):
                        print(f"🔹 NQ{start + i - NQ_START_REGISTER + 1} (Register {start + i}) = {value}")
            else:
                print(f"⚠ Failed to read NQ outputs for Registers {start} to {end - 1}")

            time.sleep(0.1)  # Small delay to avoid overloading LOGO! 8

    except Exception as e:
        print(f"❌ Exception: {e}")

    finally:
        modbus.close()
        print("🔌 Disconnected from Siemens LOGO! 8 PLC")

if __name__ == "__main__":
    main()

